/*
 * Copyright (C) 2023, Markus Theil <theil.markus@gmail.com>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include <cstdint>
#include <ctime>
#include <errno.h>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <cassert>
#include <botan/hash.h>
#include <botan/exceptn.h>

#include "esdm_crypto.h"
#include "esdm_botan.h"
#include "esdm_logger.h"
#include "ret_checkers.h"

#include "config.h"

#include <botan/stateful_rng.h>
#ifdef ESDM_BOTAN_DRNG_CHACHA20
#include <botan/chacha_rng.h>
#endif
#ifdef ESDM_BOTAN_DRNG_HMAC
#include <botan/hmac_drbg.h>
#endif

#if (defined(ESDM_BOTAN_DRNG_CHACHA20) && defined(ESDM_BOTAN_DRNG_HMAC)) ||    \
	(!defined(ESDM_BOTAN_DRNG_CHACHA20) && !defined(ESDM_BOTAN_DRNG_HMAC))
#error "Only define one Botan DRNG implementation and/or at least one!"
#endif

static const std::string DEFAULT_BOTAN_HASH{ "SHA-3(512)" };

/* introduced, as Botan only exposes unique_ptr for its digests */
struct esdm_botan_hash_ctx {
	std::unique_ptr<Botan::HashFunction> hash_fn;
};

static uint32_t esdm_botan_hash_digestsize(void *hash)
{
	(void)hash;
	return 512 / 8;
}

static int esdm_botan_hash_init(void *hash)
{
	struct esdm_botan_hash_ctx *ctx =
		reinterpret_cast<esdm_botan_hash_ctx *>(hash);

	try {
		ctx->hash_fn = Botan::HashFunction::create_or_throw(
			DEFAULT_BOTAN_HASH);
	} catch (const Botan::Lookup_Error &ex) {
		esdm_logger(LOGGER_ERR, LOGGER_C_MD,
			    "Botan::HashFunction::create() failed %s\n",
			    ex.what());
		return -EFAULT;
	}

	return 0;
}

static int esdm_botan_hash_update(void *hash, const uint8_t *inbuf,
				  size_t inbuflen)
{
	struct esdm_botan_hash_ctx *ctx =
		reinterpret_cast<esdm_botan_hash_ctx *>(hash);

	ctx->hash_fn->update(inbuf, inbuflen);

	return 0;
}

static int esdm_botan_hash_final(void *hash, uint8_t *digest)
{
	struct esdm_botan_hash_ctx *ctx =
		reinterpret_cast<esdm_botan_hash_ctx *>(hash);

	ctx->hash_fn->final(digest);

	return 0;
}

static int esdm_botan_hash_alloc(void **hash)
{
	struct esdm_botan_hash_ctx *tmp;
	struct esdm_botan_hash_ctx **ctx = (struct esdm_botan_hash_ctx **)hash;

	tmp = new struct esdm_botan_hash_ctx;
	if (!tmp)
		return -ENOMEM;

	*ctx = tmp;

	return 0;
}

static void esdm_botan_hash_dealloc(void *hash)
{
	struct esdm_botan_hash_ctx *ctx =
		reinterpret_cast<esdm_botan_hash_ctx *>(hash);

	if (ctx)
		delete ctx;
}

static const char *esdm_botan_hash_name(void)
{
	return "Botan SHA3-512";
}

static void esdm_botan_hash_desc_zero(void *hash)
{
	(void)hash;
}

struct nist_test_vector_sha3 {
	size_t len; /* in bit */
	const uint8_t *msg;
	const uint8_t *md;
};

static int esdm_botan_hash_selftest(void)
{
	/*
	 * taken from NIST FIPS 202 test vectors
	 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/sha3/sha-3bytetestvectors.zip
	 *
	 * SHA3_512ShortMsg.rsp
	 */
	static const uint8_t msg_1[] = { 0x00 };
	static const uint8_t md_1[] = {
		0xa6, 0x9f, 0x73, 0xcc, 0xa2, 0x3a, 0x9a, 0xc5, 0xc8, 0xb5,
		0x67, 0xdc, 0x18, 0x5a, 0x75, 0x6e, 0x97, 0xc9, 0x82, 0x16,
		0x4f, 0xe2, 0x58, 0x59, 0xe0, 0xd1, 0xdc, 0xc1, 0x47, 0x5c,
		0x80, 0xa6, 0x15, 0xb2, 0x12, 0x3a, 0xf1, 0xf5, 0xf9, 0x4c,
		0x11, 0xe3, 0xe9, 0x40, 0x2c, 0x3a, 0xc5, 0x58, 0xf5, 0x00,
		0x19, 0x9d, 0x95, 0xb6, 0xd3, 0xe3, 0x01, 0x75, 0x85, 0x86,
		0x28, 0x1d, 0xcd, 0x26
	};
	static const uint8_t msg_2[] = { 0x37, 0xd5, 0x18 };
	static const uint8_t md_2[] = {
		0x4a, 0xa9, 0x6b, 0x15, 0x47, 0xe6, 0x40, 0x2c, 0x0e, 0xee,
		0x78, 0x1a, 0xca, 0xa6, 0x60, 0x79, 0x7e, 0xfe, 0x26, 0xec,
		0x00, 0xb4, 0xf2, 0xe0, 0xae, 0xc4, 0xa6, 0xd1, 0x06, 0x88,
		0xdd, 0x64, 0xcb, 0xd7, 0xf1, 0x2b, 0x3b, 0x6c, 0x7f, 0x80,
		0x2e, 0x20, 0x96, 0xc0, 0x41, 0x20, 0x8b, 0x92, 0x89, 0xae,
		0xc3, 0x80, 0xd1, 0xa7, 0x48, 0xfd, 0xfc, 0xd4, 0x12, 0x85,
		0x53, 0xd7, 0x81, 0xe3
	};
	static const uint8_t msg_3[] = { 0x71, 0xa9, 0x86, 0xd2, 0xf6, 0x62 };
	static const uint8_t md_3[] = {
		0xde, 0xf6, 0xaa, 0xc2, 0xb0, 0x8c, 0x98, 0xd5, 0x6a, 0x05,
		0x01, 0xa8, 0xcb, 0x93, 0xf5, 0xb4, 0x7d, 0x63, 0x22, 0xda,
		0xf9, 0x9e, 0x03, 0x25, 0x54, 0x57, 0xc3, 0x03, 0x32, 0x63,
		0x95, 0xf7, 0x65, 0x57, 0x69, 0x30, 0xf8, 0x57, 0x1d, 0x89,
		0xc0, 0x1e, 0x72, 0x7c, 0xc7, 0x9c, 0x2d, 0x44, 0x97, 0xf8,
		0x5c, 0x45, 0x69, 0x1b, 0x55, 0x4e, 0x20, 0xda, 0x81, 0x0c,
		0x2b, 0xc8, 0x65, 0xef
	};
	static const uint8_t msg_4[] = { 0xc6, 0x1a, 0x91, 0x88, 0x81,
					 0x2a, 0xe7, 0x39, 0x94, 0xbc,
					 0x0d, 0x6d, 0x40, 0x21 };
	static const uint8_t md_4[] = {
		0x06, 0x9e, 0x6a, 0xb1, 0x67, 0x5f, 0xed, 0x8d, 0x44, 0x10,
		0x5f, 0x3b, 0x62, 0xbb, 0xf5, 0xb8, 0xff, 0x7a, 0xe8, 0x04,
		0x09, 0x89, 0x86, 0x87, 0x9b, 0x11, 0xe0, 0xd7, 0xd9, 0xb1,
		0xb4, 0xcb, 0x7b, 0xc4, 0x7a, 0xeb, 0x74, 0x20, 0x1f, 0x50,
		0x9d, 0xdc, 0x92, 0xe5, 0x63, 0x3a, 0xbd, 0x2c, 0xbe, 0x0d,
		0xdc, 0xa2, 0x48, 0x0e, 0x99, 0x08, 0xaf, 0xa6, 0x32, 0xc8,
		0xc8, 0xd5, 0xaf, 0x2a
	};

	static struct nist_test_vector_sha3 test_vectors[] = {
		{ .len = 0, .msg = msg_1, .md = md_1 },
		{ .len = 24, .msg = msg_2, .md = md_2 },
		{ .len = 48, .msg = msg_3, .md = md_3 },
		{ .len = 112, .msg = msg_4, .md = md_4 }
	};

	const size_t md_len = 512 / 8;
	uint8_t act[md_len];
	void *hash = NULL;
	int ret;

	for (size_t i = 0;
	     i < sizeof(test_vectors) / sizeof(struct nist_test_vector_sha3);
	     ++i) {
		CKINT(esdm_botan_hash_alloc(&hash));
		CKINT(esdm_botan_hash_init(hash));
		CKINT(esdm_botan_hash_update(hash, test_vectors[i].msg,
					     test_vectors[i].len / 8));
		CKINT(esdm_botan_hash_final(hash, act));

		if (memcmp(act, test_vectors[i].md, md_len)) {
			ret = -EFAULT;
			goto out;
		}

		esdm_botan_hash_dealloc(hash);
		hash = NULL;
	}

out:
	esdm_botan_hash_dealloc(hash);

	return ret;
}

const struct esdm_hash_cb esdm_botan_hash_cb = {
	.hash_name = esdm_botan_hash_name,
	.hash_selftest = esdm_botan_hash_selftest,
	.hash_digestsize = esdm_botan_hash_digestsize,
	.hash_init = esdm_botan_hash_init,
	.hash_update = esdm_botan_hash_update,
	.hash_final = esdm_botan_hash_final,
	.hash_desc_zero = esdm_botan_hash_desc_zero,
	.hash_alloc = esdm_botan_hash_alloc,
	.hash_dealloc = esdm_botan_hash_dealloc,
};

struct esdm_botan_drng_state {
	std::unique_ptr<Botan::Stateful_RNG> drbg;
	bool initialized;
};

static int esdm_botan_drbg_seed(void *drng, const uint8_t *inbuf,
				size_t inbuflen)
{
	struct esdm_botan_drng_state *state =
		reinterpret_cast<esdm_botan_drng_state *>(drng);

	if (state->initialized) {
		state->drbg->add_entropy(inbuf, inbuflen);
	} else {
#ifdef ESDM_BOTAN_DRNG_HMAC
		/* init at least with 256 bit entropy + 128 bit nonce */
		assert(inbuflen >= 3 * (256 / 8) / 2);
#endif
		state->drbg->initialize_with(inbuf, inbuflen);
		state->initialized = true;
	}

	return 0;
}

#ifdef ESDM_BOTAN_DRNG_HMAC
static ssize_t esdm_botan_drbg_generate_w_additional_data(void *drng,
							  uint8_t *outbuf,
							  size_t outbuflen,
							  const uint8_t *adata,
							  size_t adatalen)
{
	struct esdm_botan_drng_state *state =
		reinterpret_cast<esdm_botan_drng_state *>(drng);


	state->drbg->randomize_with_input(outbuf, outbuflen, adata, adatalen);

	return (ssize_t)outbuflen;
}
#endif

static ssize_t esdm_botan_drbg_generate(void *drng, uint8_t *outbuf,
					size_t outbuflen)
{
	/* calling randomize_with_input on chacha20 would
	 * trigger generating a new key on every request,
	 * skip this, as ChaCha20 should be the high speed
	 * option */
#ifdef ESDM_BOTAN_DRNG_CHACHA20
	struct esdm_botan_drng_state *state =
		reinterpret_cast<esdm_botan_drng_state *>(drng);

	state->drbg->randomize(outbuf, outbuflen);

	return (ssize_t)outbuflen;
#endif

#ifdef ESDM_BOTAN_DRNG_HMAC
	struct timespec ts;
	int ret;

	ret = clock_gettime(CLOCK_MONOTONIC, &ts);
	if (ret) {
		return -ret;
	}

	/* always use additional data in order to perform additional mixing steps
	 * inside HMAC-DRBG (recommended by BSI AIS 20/31 V3.0, Sec. 5.3.2 Par. 1079)*/
	return esdm_botan_drbg_generate_w_additional_data(
		drng, outbuf, outbuflen, reinterpret_cast<const uint8_t *>(&ts),
		sizeof(ts));
#endif
}

static void
esdm_botan_drbg_dealloc_internal(struct esdm_botan_drng_state *state)
{
	if (!state)
		return;

	state->drbg.reset();
	state->initialized = false;
}

static int esdm_botan_drbg_alloc(void **drng, uint32_t sec_strength)
{
	struct esdm_botan_drng_state *state = new struct esdm_botan_drng_state;

	(void)sec_strength;

	if (!state)
		return -ENOMEM;

#ifdef ESDM_BOTAN_DRNG_CHACHA20
	state->drbg.reset(new Botan::ChaCha_RNG());
	esdm_logger(LOGGER_VERBOSE, LOGGER_C_ANY,
		    "Botan ChaCha20 DRNG core allocated\n");
#endif
#ifdef ESDM_BOTAN_DRNG_HMAC
	state->drbg.reset(new Botan::HMAC_DRBG("SHA-512"));
	esdm_logger(LOGGER_VERBOSE, LOGGER_C_ANY,
		    "Botan SP800-90A HMAC-DRBG core allocated\n");
#endif

	*drng = state;
	if (state->drbg == nullptr) {
		esdm_botan_drbg_dealloc_internal(state);
		return -1;
	}

	return 0;
}

static void esdm_botan_drbg_dealloc(void *drng)
{
	struct esdm_botan_drng_state *state =
		reinterpret_cast<esdm_botan_drng_state *>(drng);

	esdm_botan_drbg_dealloc_internal(state);

	esdm_logger(LOGGER_VERBOSE, LOGGER_C_ANY,
		    "DRBG core zeroized and freed\n");

	delete state;
}

static const char *esdm_botan_drbg_name(void)
{
#ifdef ESDM_BOTAN_DRNG_CHACHA20
	return "Botan ChaCha20 DRNG";
#endif

#ifdef ESDM_BOTAN_DRNG_HMAC
	return "Botan SP800-90A DRBG";
#endif
}

#ifdef ESDM_BOTAN_DRNG_CHACHA20
static int esdm_botan_drbg_selftest_chacha20(void)
{
	void *drng = NULL;
	int ret;

	static const uint8_t ent_nonce[] = {
		0xBF, 0x26, 0x84, 0xC8, 0xA6, 0x9E, 0x68, 0x6E, 0xAE, 0x68,
		0x25, 0x1F, 0x33, 0x26, 0xBA, 0x4F, 0xB0, 0x82, 0x05, 0x0C,
		0x08, 0xCF, 0x26, 0x3D, 0xA6, 0x62, 0x3F, 0x4F, 0x4C, 0x44,
		0x7F, 0x71, 0xB9, 0xDE, 0xBB, 0xA0, 0xE6, 0xDD, 0x95, 0x16,
		0x4C, 0x68, 0x4C, 0x34, 0xA1, 0x77, 0x95, 0x3F, 0x98, 0xEB,
		0xC9, 0x92, 0x8E, 0x11, 0x6F, 0xBA, 0x38, 0xE3, 0xCC, 0x9C,
		0x43, 0x77, 0x9F, 0xE1
	};
	static const uint8_t reseed[] = { 0x88, 0x7F, 0x5A, 0x5C, 0xC7, 0x46,
					  0xC2, 0x9D, 0xF8, 0xD1, 0x62, 0xB4,
					  0x4F, 0x16, 0x1C, 0x77, 0x32, 0x18,
					  0xE0, 0xC2, 0xE8, 0x27, 0x60, 0x1A,
					  0x82, 0x4B, 0x0F, 0x68, 0x3C, 0x61,
					  0x40, 0x11 };

	CKINT(esdm_botan_drbg_alloc(&drng, 256));
	CKINT(esdm_botan_drbg_seed(drng, ent_nonce, sizeof(ent_nonce)));
	CKINT(esdm_botan_drbg_seed(drng, reseed, sizeof(reseed)));

	static const uint8_t exp[] = {
		0xb2, 0xe0, 0x1c, 0x33, 0xf7, 0x38, 0xd8, 0x34, 0xb9, 0xc8,
		0xf2, 0x72, 0xcf, 0x05, 0x4e, 0x8a, 0x77, 0x10, 0x93, 0x7d,
		0xa6, 0xcc, 0xeb, 0xd2, 0x94, 0x11, 0x4d, 0x51, 0x5e, 0x8f,
		0x76, 0xc4, 0x77, 0x94, 0x02, 0x5b, 0xdf, 0x55, 0x71, 0xea,
		0xd7, 0x3d, 0x9f, 0xad, 0xc0, 0x44, 0x6c, 0xc6, 0x13, 0x20,
		0x35, 0x4c, 0xa8, 0x38, 0x6f, 0x0f, 0x4c, 0x42, 0xd6, 0xb6,
		0xf1, 0x54, 0x96, 0xf2, 0xbb, 0x7c, 0xd3, 0xbe, 0xdd, 0x2f,
		0xb5, 0xcc, 0xa8, 0xb3, 0x49, 0x83, 0x1f, 0xda, 0x23, 0x0d,
		0x7a, 0x52, 0x23, 0xc8, 0xd7, 0x0b, 0x73, 0xf3, 0x3f, 0x59,
		0xde, 0xe1, 0xf1, 0x05, 0xad, 0x7b, 0x60, 0xe1, 0xaf, 0x52,
		0xd7, 0xad, 0xdc, 0xd3, 0x8a, 0x7c, 0x46, 0xd8, 0x57, 0xb9,
		0x60, 0xed, 0x2a, 0x0d, 0x6b, 0x68, 0xae, 0xdb, 0xd3, 0xe0,
		0xf4, 0xa3, 0xcf, 0x3e, 0xdb, 0x53, 0x62, 0x02,
	};
	uint8_t act[sizeof(exp)];

	if (esdm_botan_drbg_generate(drng, act, sizeof(act)) != sizeof(act)) {
		ret = -EFAULT;
		goto out;
	}

	if (esdm_botan_drbg_generate(drng, act, sizeof(act)) != sizeof(act)) {
		ret = -EFAULT;
		goto out;
	}

	if (memcmp(act, exp, sizeof(exp))) {
		ret = -EFAULT;
		goto out;
	}

out:
	esdm_botan_drbg_dealloc(drng);
	return ret;
}
#endif

#ifdef ESDM_BOTAN_DRNG_HMAC
/* we need to use test vectors with personalization strings,
 * otherwise Botan receive's too short seeds.
 * They have to be at least OutLen(HMAC_SHA512) = 512 long.
 */
struct nist_test_vector_hmac_drbg {
	const uint8_t *entropy;
	const uint8_t *nonce;
	const uint8_t *personalization_string;
	const uint8_t *returned_bits;
	const uint8_t *additional_input_1;
	const uint8_t *additional_input_2;
};

#define HMAC_TEST_VECTOR(IDX)                                                  \
	{                                                                      \
		.entropy = entropy_##IDX,                                      \
		.nonce = nonce_##IDX,                                          \
		.personalization_string = personalization_string_##IDX,        \
		.returned_bits = returned_bits_##IDX,                          \
		.additional_input_1 = nullptr,                                 \
		.additional_input_2 = nullptr,                                 \
	}

#define HMAC_TEST_VECTOR_AI(IDX)                                               \
	{                                                                      \
		.entropy = entropy_##IDX,                                      \
		.nonce = nonce_##IDX,                                          \
		.personalization_string = personalization_string_##IDX,        \
		.returned_bits = returned_bits_##IDX,                          \
		.additional_input_1 = additional_input_1_##IDX,                \
		.additional_input_2 = additional_input_2_##IDX,                \
	}

static int esdm_botan_drbg_selftest_hmac()
{
	void *drng = NULL;
	uint8_t *seed_material;
	uint8_t *act;
	int ret;

	/*
	 * Taken from NIST SP800-90A DRBG Test Vectors
	 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/drbg/drbgtestvectors.zip
	 * -> drbgvectors_no_reseed.zip -> SHA-512 with personalization string
	 */
	const uint8_t entropy_1[] = { 0xf8, 0x56, 0x49, 0xf8, 0xa8, 0xc0, 0x1f,
				      0x8a, 0x29, 0x25, 0xf7, 0xe9, 0x3e, 0x35,
				      0x6b, 0xe0, 0xfb, 0xc7, 0x52, 0x06, 0xd1,
				      0xce, 0x2f, 0x7e, 0x04, 0xa1, 0x42, 0x57,
				      0xc3, 0x33, 0x8a, 0x48 };
	const uint8_t nonce_1[] = { 0x01, 0xa8, 0x84, 0x3b, 0x7a, 0x3d,
				    0xc0, 0xb1, 0xca, 0xb5, 0xf7, 0xc8,
				    0xb0, 0x01, 0x59, 0x8d };
	const uint8_t personalization_string_1[] = {
		0x91, 0x02, 0x2d, 0x07, 0x28, 0x24, 0xcb, 0x58,
		0x0a, 0xc0, 0x75, 0x55, 0xc9, 0x0a, 0x31, 0x37,
		0x3f, 0x2d, 0xfc, 0x27, 0x2d, 0xe6, 0x0d, 0x3b,
		0xdb, 0xc0, 0x61, 0x2e, 0x17, 0x58, 0x74, 0xab
	};
	const uint8_t returned_bits_1[] = {
		0x9f, 0x40, 0xaf, 0xfd, 0xa9, 0xc7, 0x97, 0x36, 0x3a, 0x7b,
		0x05, 0x8d, 0x7f, 0x5b, 0x84, 0x8f, 0x0b, 0xa3, 0x66, 0x80,
		0x8e, 0x7a, 0x87, 0xec, 0x94, 0x89, 0x40, 0x91, 0xca, 0x08,
		0x18, 0xdc, 0x38, 0x7a, 0x8a, 0xb2, 0x49, 0xaf, 0x32, 0x68,
		0xa9, 0xb0, 0xc9, 0x5f, 0x7a, 0x78, 0x1d, 0x3c, 0x7b, 0xa6,
		0x13, 0x46, 0x0a, 0x2a, 0x5f, 0x8e, 0xe4, 0xbf, 0x9e, 0xd2,
		0x37, 0xaf, 0xae, 0xf6, 0x9f, 0x04, 0x6e, 0x16, 0x37, 0x99,
		0x79, 0x42, 0x52, 0xd2, 0xc9, 0x70, 0x8e, 0xa3, 0x37, 0xb8,
		0xbc, 0xc6, 0x74, 0xc5, 0x01, 0x61, 0x72, 0xc5, 0x0d, 0x6d,
		0xe9, 0xd9, 0x41, 0xf4, 0x96, 0xcb, 0xde, 0x4a, 0x4e, 0x7a,
		0xbc, 0xdf, 0xd4, 0x2a, 0x98, 0x91, 0x19, 0x22, 0x4e, 0xa7,
		0xf3, 0xc4, 0x72, 0x71, 0x79, 0xf4, 0x2b, 0xbb, 0x54, 0x46,
		0x6f, 0x53, 0x0e, 0x43, 0x94, 0xcf, 0x18, 0xc1, 0x5b, 0x54,
		0x8c, 0xa6, 0x10, 0x51, 0x18, 0x00, 0xb3, 0x9d, 0xa6, 0x92,
		0x97, 0xa4, 0x6e, 0xd1, 0xb2, 0x37, 0x72, 0x2c, 0x6d, 0x50,
		0x24, 0x82, 0x45, 0xf7, 0xe9, 0x0f, 0x34, 0x36, 0x17, 0xac,
		0xa4, 0x4b, 0x62, 0x45, 0xc9, 0x9a, 0x71, 0x4f, 0x71, 0x9e,
		0x32, 0x4c, 0x2f, 0xd4, 0xb9, 0x5e, 0x84, 0xb3, 0xf7, 0xd3,
		0x21, 0x29, 0x86, 0xdd, 0x7f, 0x51, 0xf6, 0x0e, 0x38, 0xd1,
		0xa0, 0x77, 0x39, 0x8f, 0x5a, 0xf3, 0x1a, 0xda, 0xe8, 0x64,
		0x7b, 0xc9, 0x20, 0xe6, 0xa0, 0xc8, 0x65, 0xa2, 0x97, 0xbf,
		0xfc, 0x88, 0xb4, 0x07, 0xa2, 0x4e, 0x2d, 0x2a, 0xea, 0xd0,
		0xe7, 0x11, 0xfc, 0xf9, 0x6b, 0x83, 0xbb, 0xed, 0xe5, 0xbb,
		0x35, 0xa5, 0xf9, 0xb0, 0xdc, 0x5e, 0x23, 0x76, 0xa6, 0x93,
		0xaf, 0xa9, 0x07, 0x0a, 0x4b, 0x1a, 0xa8, 0x6d, 0xec, 0x23,
		0xe6, 0xe5, 0x97, 0xa7, 0x42, 0x37
	};
	const uint8_t entropy_2[] = { 0x58, 0xb8, 0x10, 0x01, 0xb7, 0x6e, 0x24,
				      0x8d, 0x9a, 0x4e, 0xab, 0x5e, 0x73, 0x71,
				      0xa9, 0x7d, 0xd0, 0x99, 0x2b, 0xd4, 0xed,
				      0x6d, 0xe1, 0x80, 0x84, 0xb2, 0x32, 0x1b,
				      0x8c, 0x29, 0x2b, 0x30 };
	const uint8_t nonce_2[] = { 0xd4, 0xe9, 0xdf, 0x17, 0x1d, 0x09,
				    0xa7, 0xb9, 0x39, 0x49, 0x90, 0x7c,
				    0xa6, 0x5f, 0xd5, 0xb0 };
	const uint8_t personalization_string_2[] = {
		0xa3, 0x6d, 0x99, 0x6e, 0xcf, 0x48, 0x05, 0x8a,
		0xc8, 0x50, 0xd6, 0x6e, 0x56, 0xd8, 0x12, 0xe8,
		0xc5, 0x8e, 0xc0, 0xd8, 0x9a, 0x30, 0xdc, 0xf3,
		0x04, 0x75, 0xd3, 0x83, 0x0c, 0x4a, 0x33, 0xc4
	};
	const uint8_t returned_bits_2[] = {
		0x22, 0xd8, 0xe1, 0xf5, 0x0c, 0x30, 0x12, 0x08, 0xc5, 0xf1,
		0x98, 0xf2, 0x9c, 0xf9, 0xf9, 0x77, 0x04, 0xf2, 0xdf, 0x46,
		0xf6, 0x62, 0xbf, 0xd7, 0xbd, 0x28, 0x83, 0xc2, 0x2b, 0x49,
		0xab, 0x5d, 0x7b, 0xed, 0x1a, 0x2b, 0x6c, 0xf0, 0xc5, 0x5a,
		0x0a, 0xa0, 0xb6, 0x95, 0xc5, 0x26, 0x95, 0xb5, 0x79, 0x19,
		0xdb, 0xbe, 0x83, 0x81, 0xfc, 0x56, 0xe7, 0x25, 0x53, 0x15,
		0x8e, 0x10, 0xab, 0x59, 0xd4, 0xe6, 0x82, 0xdf, 0xc3, 0xa0,
		0x71, 0x99, 0x87, 0x6c, 0x26, 0x6d, 0xcb, 0xe0, 0x77, 0x35,
		0x38, 0x2d, 0x8e, 0xc0, 0x79, 0xab, 0x4e, 0xf5, 0xe6, 0x22,
		0xa2, 0x32, 0x90, 0x92, 0xcd, 0xe4, 0x5c, 0x53, 0xf7, 0x1a,
		0xeb, 0xe4, 0x3b, 0x34, 0x99, 0x0f, 0x55, 0xad, 0x03, 0x7e,
		0xb3, 0x3a, 0x87, 0xab, 0xbc, 0xd3, 0xad, 0xa3, 0x11, 0x2e,
		0xc4, 0x3b, 0x9b, 0xb2, 0x0b, 0x61, 0xba, 0xc5, 0x49, 0x73,
		0x32, 0x22, 0x31, 0x3d, 0xe2, 0xbd, 0x91, 0xd5, 0x32, 0xf1,
		0xc1, 0x7b, 0xc3, 0xde, 0x3f, 0xa3, 0x4f, 0x7d, 0x3a, 0xeb,
		0xb2, 0xb5, 0x9d, 0x6c, 0x8b, 0x90, 0xce, 0x21, 0x1e, 0x18,
		0x4f, 0xbe, 0x54, 0xac, 0x0c, 0xb9, 0x7d, 0x7a, 0x03, 0xe0,
		0x57, 0xa6, 0xbf, 0x69, 0xc4, 0xde, 0xb4, 0xf1, 0x1c, 0xab,
		0x49, 0xbc, 0xef, 0x32, 0x3e, 0x9c, 0xa2, 0xec, 0xa1, 0xf4,
		0xde, 0xc5, 0x0a, 0x02, 0xae, 0x1d, 0xee, 0xa1, 0xd5, 0x6f,
		0x03, 0x7d, 0x53, 0xda, 0x03, 0xda, 0x87, 0x0d, 0xde, 0x13,
		0xac, 0x7a, 0x08, 0x4d, 0x1e, 0x2a, 0xa4, 0xf0, 0x95, 0xb3,
		0xcb, 0xb1, 0x27, 0x6a, 0x4f, 0x58, 0x15, 0xcd, 0xe0, 0x3a,
		0x1e, 0xad, 0x7d, 0x5f, 0xf8, 0x7f, 0xe7, 0x45, 0x3c, 0xc6,
		0x03, 0xc0, 0xa1, 0xd1, 0xf6, 0xaf, 0x1e, 0xae, 0xde, 0xc3,
		0x58, 0x35, 0xa6, 0xce, 0x0c, 0x9b
	};
	const uint8_t entropy_3[] = { 0xaa, 0x1c, 0xfe, 0x4c, 0x9f, 0x03, 0xa9,
				      0x99, 0x0c, 0x2d, 0xd5, 0x5c, 0x07, 0x75,
				      0xec, 0x36, 0xf3, 0xda, 0xf3, 0xd6, 0x48,
				      0x73, 0xef, 0x84, 0x36, 0x92, 0xb5, 0x88,
				      0x7b, 0x7d, 0x7a, 0x4c };
	const uint8_t nonce_3[] = { 0x70, 0x48, 0x47, 0x96, 0xff, 0xf9,
				    0x0a, 0x01, 0xcf, 0xc2, 0x16, 0x8d,
				    0x1d, 0x0f, 0x5e, 0xd9 };
	const uint8_t personalization_string_3[] = {
		0x21, 0xb4, 0x68, 0xff, 0x6f, 0x1e, 0x92, 0xca,
		0x41, 0x01, 0x77, 0x94, 0x1e, 0xc0, 0xbe, 0xa7,
		0x49, 0x90, 0x55, 0x50, 0x71, 0xcb, 0x7d, 0x7f,
		0xc0, 0x84, 0x4f, 0xb2, 0x4c, 0xef, 0xd5, 0x11
	};
	const uint8_t returned_bits_3[] = {
		0x72, 0xca, 0x74, 0x67, 0x1d, 0x10, 0xf6, 0xaa, 0x06, 0x07,
		0xc3, 0xab, 0xf4, 0xe8, 0x6f, 0x02, 0x2a, 0xf3, 0x4c, 0xd8,
		0xc3, 0x2c, 0xbc, 0x27, 0x4e, 0xff, 0x9b, 0xd3, 0x8b, 0xcd,
		0x4b, 0x79, 0xc6, 0x4b, 0x09, 0xe6, 0x8f, 0x44, 0xa3, 0xb1,
		0xc6, 0x59, 0x2b, 0x7f, 0x45, 0x8f, 0x74, 0x3f, 0x99, 0xa3,
		0x97, 0x1b, 0xec, 0xf6, 0xac, 0xd9, 0x74, 0x79, 0x2d, 0xd5,
		0x81, 0x2d, 0x47, 0x38, 0x91, 0x76, 0x64, 0x60, 0xb0, 0x28,
		0xf3, 0x72, 0x40, 0x36, 0x49, 0x61, 0x88, 0xbf, 0xd2, 0xfa,
		0xe5, 0xd3, 0xbf, 0x73, 0x4d, 0x25, 0x03, 0x31, 0x6d, 0xb5,
		0xc9, 0xb4, 0x08, 0x49, 0x6e, 0x9d, 0x99, 0x71, 0x2d, 0xec,
		0x35, 0x58, 0x65, 0xc2, 0x40, 0xd3, 0x20, 0x9c, 0x17, 0x93,
		0xfc, 0xc8, 0x15, 0xdf, 0x05, 0x7f, 0xfe, 0x55, 0x5d, 0x2b,
		0xbf, 0x47, 0xe7, 0x37, 0x00, 0x31, 0x89, 0x3a, 0xef, 0xec,
		0x6e, 0xf0, 0xe7, 0x1d, 0x13, 0x7e, 0x16, 0xc4, 0x16, 0xdb,
		0xe5, 0x2d, 0xf3, 0xb9, 0x8f, 0x28, 0x0f, 0x5e, 0x6b, 0x29,
		0x63, 0x6d, 0x7d, 0x9b, 0xb0, 0x99, 0x91, 0x39, 0xfb, 0x5e,
		0x46, 0xbf, 0xec, 0x53, 0xbb, 0x22, 0xc8, 0xfd, 0x68, 0xd2,
		0xa1, 0x5d, 0xf7, 0xd4, 0xe1, 0xa2, 0xc8, 0x13, 0xae, 0x36,
		0x3b, 0xf3, 0x05, 0x5e, 0x6d, 0xef, 0x00, 0x8a, 0xe1, 0x63,
		0x43, 0xde, 0xd5, 0x98, 0xa0, 0xc0, 0x85, 0x48, 0x80, 0x11,
		0x89, 0xce, 0xd0, 0x3e, 0x6f, 0x81, 0xb3, 0x12, 0x86, 0xdd,
		0x8e, 0x3e, 0xab, 0x82, 0xd6, 0x48, 0xb6, 0xa9, 0xfe, 0x5a,
		0xa6, 0xfa, 0x1f, 0x95, 0x17, 0x89, 0xcf, 0x4f, 0x58, 0x15,
		0xe1, 0x01, 0x88, 0xd2, 0xd9, 0x75, 0x5e, 0x04, 0xf2, 0x83,
		0x37, 0x3c, 0xe7, 0xda, 0x80, 0x6c, 0xd9, 0x29, 0x35, 0xae,
		0xa2, 0x43, 0xd6, 0xa5, 0x66, 0xf9
	};
	const uint8_t entropy_4[] = { 0x26, 0x25, 0x40, 0x0a, 0x61, 0x30, 0xe7,
				      0x7f, 0xa2, 0x0f, 0x0b, 0x7c, 0x58, 0xcc,
				      0x0f, 0x9d, 0x0f, 0x22, 0xef, 0x5e, 0xa5,
				      0x4e, 0x5c, 0xfa, 0x6e, 0x6e, 0x64, 0x89,
				      0x33, 0xd6, 0x23, 0x22 };
	const uint8_t nonce_4[] = { 0xdf, 0xf0, 0xeb, 0x02, 0xfc, 0x7c,
				    0xe9, 0x3e, 0x4b, 0xe2, 0xe2, 0x94,
				    0xa2, 0x45, 0x09, 0xb4 };
	const uint8_t personalization_string_4[] = {
		0xcd, 0xb1, 0x07, 0x2f, 0xd8, 0x31, 0x14, 0xef,
		0x83, 0x32, 0x3c, 0xcb, 0x03, 0xd7, 0x02, 0xf6,
		0x8a, 0x55, 0x0d, 0x7a, 0x0e, 0x8b, 0x15, 0x3a,
		0x95, 0xeb, 0x9f, 0x77, 0x1a, 0x27, 0x18, 0x61
	};
	const uint8_t returned_bits_4[] = {
		0x4f, 0xa6, 0x07, 0x17, 0x56, 0xaf, 0xe9, 0x72, 0x85, 0x30,
		0x72, 0xba, 0x73, 0xe2, 0x03, 0x77, 0x39, 0x37, 0xf7, 0xd6,
		0xe1, 0xdc, 0xd7, 0x98, 0x2c, 0x37, 0xef, 0x75, 0x66, 0x36,
		0x9e, 0x27, 0xad, 0xd7, 0x88, 0x9b, 0x32, 0xed, 0x2e, 0x2c,
		0x26, 0x69, 0xd0, 0x06, 0x33, 0xc2, 0x2d, 0x91, 0xfd, 0x98,
		0xf2, 0x22, 0x37, 0x89, 0x68, 0x1f, 0xd3, 0xb1, 0xd9, 0x3a,
		0x16, 0xc2, 0x27, 0x8b, 0x8e, 0x40, 0x07, 0x28, 0x6d, 0xc1,
		0x27, 0x3e, 0x5a, 0xbd, 0xce, 0x78, 0x45, 0xf8, 0x7c, 0x06,
		0x6a, 0x59, 0xb0, 0xd8, 0x15, 0x2f, 0x9c, 0x13, 0x23, 0x0c,
		0xe4, 0x88, 0xf9, 0xcf, 0x00, 0xe7, 0xc3, 0xe1, 0xe0, 0xc3,
		0x71, 0x9e, 0x60, 0x87, 0xfa, 0x25, 0x1e, 0x8b, 0xb4, 0xdd,
		0xd3, 0xe7, 0x78, 0xc6, 0x14, 0x9a, 0x14, 0xdb, 0x32, 0x17,
		0xf1, 0x72, 0xa8, 0xbf, 0xd6, 0x56, 0xbf, 0xe6, 0xfe, 0x2c,
		0x49, 0x60, 0x76, 0x8a, 0xde, 0xca, 0x84, 0x3f, 0x1f, 0x36,
		0x67, 0x1b, 0x72, 0x2f, 0xcc, 0x62, 0xdd, 0xe8, 0xb2, 0xe0,
		0xc5, 0x76, 0x5c, 0x62, 0x32, 0xad, 0xa6, 0xa4, 0x50, 0x4b,
		0x21, 0xf3, 0x50, 0x78, 0x6d, 0x08, 0x79, 0x40, 0x8d, 0x89,
		0x79, 0xe1, 0x9a, 0x3a, 0xa1, 0x6f, 0x0e, 0xc4, 0x7d, 0xf3,
		0xd0, 0x83, 0x62, 0x96, 0xfb, 0x88, 0x36, 0x3d, 0x1f, 0xbc,
		0x68, 0x7b, 0xed, 0x75, 0xf1, 0x4c, 0xcc, 0xea, 0xfb, 0x18,
		0x5c, 0xe9, 0x0c, 0x2b, 0x0f, 0x5c, 0xef, 0x32, 0x47, 0xf1,
		0x57, 0x3e, 0x82, 0x2c, 0x19, 0xb1, 0xb9, 0x94, 0x42, 0x04,
		0xd4, 0x2a, 0x61, 0x01, 0x32, 0x6b, 0x12, 0x72, 0x62, 0xb7,
		0x02, 0x3e, 0x89, 0x82, 0x4a, 0x3f, 0x2d, 0x4f, 0xbc, 0x4c,
		0x07, 0x2e, 0xd3, 0xdb, 0xb7, 0x32, 0x5e, 0x77, 0xec, 0xcc,
		0x70, 0xcd, 0x82, 0x21, 0x8f, 0x43
	};
	const uint8_t entropy_5[] = { 0x49, 0x43, 0xe0, 0xc7, 0x25, 0xcb, 0xa7,
				      0xac, 0x3b, 0xba, 0x16, 0xd1, 0xde, 0x34,
				      0x44, 0xeb, 0x36, 0x71, 0x29, 0xfa, 0x38,
				      0xd1, 0xbd, 0x6e, 0x00, 0x32, 0xec, 0x1b,
				      0x3e, 0x0b, 0x76, 0x7c };
	const uint8_t nonce_5[] = { 0x74, 0x37, 0x75, 0xb8, 0x79, 0xb2,
				    0x2b, 0xa2, 0x4d, 0x0b, 0x08, 0xde,
				    0xd6, 0x29, 0x8e, 0xf0 };
	const uint8_t personalization_string_5[] = {
		0x1d, 0x8d, 0x1a, 0xbc, 0x5f, 0x59, 0x76, 0x48,
		0x88, 0x9b, 0xc2, 0x20, 0x04, 0x0c, 0xfe, 0x3b,
		0xd4, 0x9a, 0xd7, 0x71, 0xeb, 0xf7, 0xff, 0x2f,
		0xfc, 0x9a, 0x7b, 0xb8, 0xce, 0x2c, 0xb2, 0x85
	};
	const uint8_t returned_bits_5[] = {
		0x1c, 0xd4, 0x5a, 0x83, 0xf0, 0x23, 0x6d, 0x70, 0x90, 0x82,
		0x91, 0xa0, 0xcb, 0x24, 0xe8, 0xaf, 0x67, 0x26, 0x8b, 0xa1,
		0x44, 0x92, 0xc0, 0xee, 0x4f, 0x7f, 0x73, 0x6e, 0x48, 0xa3,
		0xda, 0x7c, 0x5b, 0xc8, 0x0a, 0x0a, 0xaf, 0xb5, 0x4a, 0xb0,
		0x28, 0x90, 0x74, 0x77, 0x46, 0x3d, 0x75, 0x87, 0xa1, 0x1c,
		0x59, 0xcf, 0x2a, 0xdd, 0x25, 0xaa, 0x79, 0x9f, 0xc3, 0x92,
		0x90, 0xc2, 0x28, 0x4d, 0xa9, 0x16, 0x79, 0x09, 0x85, 0xdd,
		0x1d, 0x84, 0x24, 0x21, 0xa2, 0x41, 0x86, 0xae, 0xc5, 0xac,
		0xad, 0x4e, 0xd5, 0xe5, 0x11, 0xda, 0xb8, 0xc2, 0x8f, 0x2a,
		0x2a, 0x0b, 0x24, 0x49, 0xd1, 0xaa, 0xeb, 0x52, 0x65, 0xe5,
		0xdf, 0x2e, 0x70, 0x8e, 0x76, 0x1a, 0x42, 0xdf, 0x53, 0x04,
		0xdc, 0x92, 0xdd, 0x2c, 0x10, 0x02, 0x8c, 0xbd, 0xab, 0xf5,
		0xf2, 0x1c, 0x1c, 0xf5, 0x72, 0xcb, 0x0d, 0x55, 0xc1, 0x94,
		0xa2, 0x73, 0xc1, 0x66, 0xdf, 0x5d, 0xce, 0x46, 0x9d, 0xda,
		0xef, 0x53, 0x7b, 0x3f, 0x5e, 0x72, 0xb6, 0xec, 0x8c, 0x3f,
		0x32, 0xe2, 0xa6, 0x9d, 0x90, 0xa6, 0x40, 0xed, 0x3f, 0x21,
		0x98, 0x32, 0x11, 0x76, 0x15, 0xc2, 0x09, 0x25, 0x1a, 0x46,
		0x7f, 0x73, 0x43, 0xdf, 0x9a, 0x6f, 0xb8, 0x91, 0xc6, 0x64,
		0x6f, 0x01, 0xf8, 0x96, 0xd8, 0x33, 0x19, 0xbe, 0xc6, 0x3c,
		0xfb, 0x0e, 0x36, 0xd3, 0x5d, 0x5b, 0x55, 0x4f, 0x08, 0xa0,
		0x63, 0x5c, 0x36, 0x05, 0xd5, 0x1f, 0x9e, 0x7c, 0xb8, 0xc3,
		0x2c, 0xf6, 0x9c, 0xf3, 0xcb, 0xdd, 0xb0, 0x2f, 0x68, 0xc3,
		0x21, 0xae, 0x08, 0xcd, 0x18, 0x5d, 0xe2, 0x02, 0x56, 0xc9,
		0xc7, 0x8e, 0xfd, 0x66, 0x7a, 0x35, 0xa3, 0xc4, 0x23, 0xe6,
		0xa0, 0xd8, 0x32, 0x13, 0xfe, 0x0a, 0xcc, 0x1a, 0x11, 0x69,
		0x3b, 0x64, 0x10, 0x37, 0xda, 0xec
	};
	const uint8_t entropy_6[] = { 0xc9, 0x0e, 0xf2, 0xa9, 0xa1, 0xcf, 0x49,
				      0x36, 0x68, 0x9e, 0xc6, 0x6a, 0xcf, 0x77,
				      0x1e, 0x3b, 0x0f, 0xaa, 0xab, 0xba, 0xe4,
				      0xaf, 0x86, 0x6d, 0x45, 0x95, 0xd2, 0x08,
				      0x23, 0xd0, 0x4d, 0xf7 };
	const uint8_t nonce_6[] = { 0x62, 0x4c, 0xaf, 0x43, 0xc3, 0xfb,
				    0x18, 0xb0, 0xa9, 0xb9, 0xb7, 0xb5,
				    0x62, 0x45, 0xce, 0x60 };
	const uint8_t personalization_string_6[] = {
		0xef, 0x05, 0xd1, 0x46, 0x50, 0x11, 0x40, 0x38,
		0xc3, 0x0f, 0x44, 0x1f, 0x84, 0x84, 0x30, 0x1d,
		0x24, 0x42, 0xf6, 0xb9, 0x16, 0x7f, 0x57, 0xf1,
		0xf2, 0xc0, 0xa7, 0x7b, 0x5c, 0x91, 0x21, 0x2a
	};
	const uint8_t returned_bits_6[] = {
		0xf8, 0x97, 0x63, 0xa9, 0x6e, 0xbb, 0xc1, 0xc4, 0xb5, 0xd5,
		0x73, 0x05, 0x01, 0xdd, 0x5a, 0x59, 0x10, 0xa9, 0xb9, 0x07,
		0xb1, 0x93, 0x94, 0x42, 0x56, 0x2e, 0xd8, 0x80, 0xbc, 0xa9,
		0x47, 0x24, 0xf9, 0x39, 0x2d, 0x31, 0xec, 0xed, 0x9b, 0xb6,
		0x70, 0xcf, 0x35, 0x5d, 0x51, 0x13, 0xa7, 0x8f, 0x6b, 0x69,
		0x93, 0xa8, 0x59, 0x54, 0xc8, 0xe5, 0x2d, 0x45, 0x0f, 0x14,
		0x81, 0x5e, 0x8d, 0x25, 0xf3, 0xc5, 0xaf, 0x5e, 0x07, 0x91,
		0xfc, 0x9f, 0x94, 0x22, 0x09, 0x94, 0xd5, 0x8b, 0x7e, 0x3d,
		0xec, 0xda, 0xa0, 0x19, 0x93, 0x42, 0x8b, 0xb5, 0x97, 0xf1,
		0x12, 0x9c, 0x7c, 0x9a, 0x06, 0x43, 0xc4, 0x13, 0xd0, 0x69,
		0x6d, 0xf1, 0x91, 0xab, 0x0b, 0x18, 0x69, 0x1c, 0x77, 0xe7,
		0x9d, 0x57, 0x05, 0xed, 0x14, 0x4a, 0x3b, 0x54, 0xcc, 0xb3,
		0xe0, 0x25, 0xf4, 0xf7, 0x1d, 0x62, 0xcb, 0xba, 0xb9, 0xac,
		0xa0, 0x80, 0x4b, 0x15, 0x32, 0x5d, 0x33, 0x5a, 0x35, 0xb8,
		0xad, 0x9d, 0x06, 0x43, 0x5c, 0xa2, 0xb8, 0x47, 0xfc, 0xf4,
		0x5a, 0x2b, 0xd0, 0xaf, 0xc9, 0x55, 0x83, 0xcc, 0x11, 0xdd,
		0x66, 0xf6, 0x35, 0x96, 0xc9, 0xa8, 0x1e, 0xc1, 0x24, 0x4a,
		0x58, 0xae, 0xe3, 0x52, 0xb5, 0x9c, 0xd4, 0x17, 0x3e, 0x1b,
		0x5b, 0x51, 0xe8, 0x8f, 0x1d, 0x64, 0xed, 0x88, 0x7d, 0xc7,
		0xc2, 0x3e, 0xfb, 0xf7, 0xef, 0x49, 0xe3, 0x4b, 0xc1, 0x32,
		0x4e, 0x91, 0xd1, 0x74, 0x03, 0x05, 0x27, 0x41, 0xdc, 0xd7,
		0xe1, 0xe5, 0xa8, 0x66, 0xce, 0x3e, 0x2e, 0x42, 0x40, 0x96,
		0xa4, 0xc6, 0x7a, 0xd8, 0x18, 0x22, 0x0a, 0x36, 0xa4, 0xef,
		0x83, 0x79, 0xf7, 0x65, 0x0a, 0x2e, 0x49, 0xa9, 0x53, 0xfe,
		0x48, 0x82, 0x02, 0xf1, 0xa9, 0x5f, 0xb5, 0x5a, 0x92, 0x9b,
		0x0e, 0x32, 0x61, 0xf1, 0x81, 0xb7
	};
	const uint8_t entropy_7[] = { 0xfc, 0x9e, 0x6a, 0x14, 0xf9, 0x61, 0x4e,
				      0x14, 0x30, 0xdf, 0xc9, 0x96, 0x40, 0x63,
				      0x3f, 0x21, 0x95, 0xe3, 0x55, 0x32, 0x34,
				      0x2c, 0xce, 0xde, 0x22, 0x6e, 0x08, 0x96,
				      0xd0, 0xc0, 0x73, 0x19 };
	const uint8_t nonce_7[] = { 0xa4, 0x3c, 0x19, 0x08, 0x29, 0x88,
				    0xa2, 0xa1, 0x60, 0x91, 0x50, 0xd9,
				    0x03, 0x10, 0x64, 0x15 };
	const uint8_t personalization_string_7[] = {
		0xec, 0x6f, 0x5a, 0xea, 0x1c, 0xff, 0x91, 0xb0,
		0xb7, 0x98, 0xb1, 0x24, 0x7a, 0x2e, 0x79, 0xbd,
		0x62, 0x17, 0x3d, 0x8c, 0xcd, 0xb5, 0xda, 0xf9,
		0x49, 0x2b, 0x39, 0x58, 0xad, 0x82, 0xc4, 0x8f
	};
	const uint8_t returned_bits_7[] = {
		0x85, 0x2a, 0x9d, 0x86, 0x45, 0x04, 0x84, 0x6f, 0xda, 0x05,
		0x67, 0x7c, 0x56, 0x7b, 0xd9, 0xff, 0xf9, 0x52, 0x21, 0xe6,
		0x20, 0xb8, 0x47, 0x3c, 0x5e, 0x51, 0x58, 0x06, 0x97, 0xae,
		0x19, 0x3d, 0x7b, 0xf1, 0xf3, 0x23, 0x7f, 0xd0, 0x94, 0x36,
		0x3b, 0xef, 0x04, 0xf9, 0x23, 0x1d, 0x95, 0xb8, 0x53, 0x1a,
		0x83, 0x7a, 0xd4, 0x3c, 0x7f, 0xf1, 0x8d, 0x16, 0x5a, 0xcb,
		0x25, 0x02, 0xf9, 0xe6, 0x45, 0x46, 0x0e, 0x06, 0xe2, 0xda,
		0x19, 0xbd, 0x10, 0xc2, 0xdf, 0x17, 0xef, 0x2a, 0x6c, 0x07,
		0x91, 0x92, 0x1d, 0x74, 0xf7, 0x2f, 0x91, 0xdb, 0x4f, 0x80,
		0x48, 0x4f, 0x4f, 0xe3, 0xeb, 0x0a, 0xed, 0xac, 0x4e, 0x64,
		0x17, 0xce, 0xf7, 0x6e, 0x75, 0xc6, 0x8c, 0x04, 0x1b, 0xe2,
		0x90, 0xf9, 0x55, 0xa6, 0x12, 0x14, 0x99, 0x84, 0x25, 0x49,
		0xea, 0x92, 0x51, 0xbf, 0x89, 0x21, 0x07, 0x61, 0x4a, 0x1e,
		0xe9, 0x41, 0x10, 0xca, 0x40, 0x1c, 0x09, 0x6c, 0x5a, 0x88,
		0xad, 0xc4, 0xb3, 0x9b, 0xfa, 0x5f, 0x82, 0x1f, 0xb9, 0x80,
		0xb6, 0xe6, 0x45, 0x5e, 0xe7, 0x5c, 0xfd, 0xb8, 0xcf, 0xb3,
		0x17, 0x56, 0x56, 0x5c, 0x3c, 0x83, 0x6c, 0xc8, 0xe9, 0x6b,
		0x53, 0x70, 0x80, 0xfa, 0xcd, 0x6c, 0x4c, 0x38, 0xc5, 0xd1,
		0x5c, 0x40, 0x5e, 0x90, 0x4a, 0x9d, 0x70, 0x8e, 0x92, 0xa6,
		0x26, 0xce, 0xc6, 0x3a, 0x5f, 0x88, 0xe3, 0xfb, 0x2b, 0x59,
		0x73, 0x48, 0xa7, 0x79, 0x85, 0x59, 0xd6, 0x12, 0xfc, 0xe9,
		0x01, 0x3e, 0x6c, 0xda, 0x53, 0xb8, 0xee, 0xf6, 0x1d, 0x5c,
		0x87, 0x4e, 0xa0, 0x30, 0xd3, 0x77, 0x63, 0x9f, 0x93, 0xf3,
		0xc8, 0x3b, 0x5f, 0x7f, 0x63, 0x4d, 0xa2, 0x01, 0xce, 0x39,
		0x2d, 0x97, 0x99, 0x10, 0x2b, 0xa9, 0xca, 0x4c, 0x61, 0x26,
		0xdd, 0x9f, 0xb7, 0x7b, 0x33, 0xe1
	};
	const uint8_t entropy_8[] = { 0xa5, 0xe0, 0xb9, 0x4b, 0x0e, 0xfb, 0x6e,
				      0x1d, 0xf2, 0xb6, 0x91, 0xbd, 0xd3, 0xe1,
				      0x5d, 0x68, 0xba, 0x60, 0x2e, 0x25, 0xa4,
				      0xe1, 0xfa, 0x7b, 0x17, 0xfe, 0x12, 0x4f,
				      0x28, 0x0b, 0x72, 0x2b };
	const uint8_t nonce_8[] = { 0x3b, 0xe2, 0x66, 0x57, 0xfd, 0x44,
				    0x36, 0x8e, 0xfe, 0xbe, 0x93, 0xc1,
				    0x6b, 0x23, 0x0e, 0x4f };
	const uint8_t personalization_string_8[] = {
		0x65, 0x2f, 0xce, 0x43, 0x40, 0x71, 0x46, 0xc0,
		0x69, 0xc6, 0xd2, 0x0e, 0x55, 0x7e, 0xb7, 0x5b,
		0xbb, 0x7c, 0x72, 0xc1, 0x0a, 0xfe, 0xaf, 0xfb,
		0xdb, 0x9f, 0xf4, 0xbb, 0x6b, 0x2d, 0x05, 0x58
	};
	const uint8_t returned_bits_8[] = {
		0xbc, 0x93, 0x79, 0xfc, 0xa8, 0x50, 0x7d, 0x56, 0x24, 0x34,
		0xc8, 0xd5, 0x4d, 0x35, 0xdf, 0xec, 0xe7, 0x2c, 0x18, 0x19,
		0x75, 0x93, 0x48, 0x0b, 0xfa, 0x7b, 0x63, 0xb2, 0x62, 0xd7,
		0x96, 0x67, 0xd8, 0x18, 0x4c, 0x90, 0x26, 0x68, 0x2b, 0x4a,
		0x42, 0xb7, 0x3e, 0xe2, 0x37, 0xce, 0x5f, 0xf4, 0x6d, 0x8c,
		0xf5, 0x8d, 0xf4, 0xcd, 0x7f, 0x14, 0x5b, 0xa4, 0xae, 0x39,
		0x64, 0x83, 0x8a, 0x78, 0x7f, 0x35, 0x3f, 0x82, 0xc7, 0xba,
		0x02, 0xf9, 0x68, 0x0b, 0x0b, 0x72, 0xbd, 0x37, 0x47, 0x3d,
		0xc9, 0x8f, 0xea, 0x00, 0x4b, 0xa5, 0xf3, 0xb8, 0x7c, 0x19,
		0x31, 0x70, 0xf0, 0x0a, 0xba, 0x61, 0xd1, 0x6e, 0xca, 0x61,
		0x2e, 0xfc, 0x10, 0x1c, 0x9d, 0xb9, 0x16, 0xd9, 0xad, 0x9f,
		0x34, 0xc2, 0x05, 0x95, 0xa4, 0xdf, 0x87, 0x10, 0x6b, 0x95,
		0x2b, 0x9d, 0x3d, 0x47, 0xf3, 0xee, 0x34, 0x6f, 0x15, 0x0e,
		0x8e, 0x4e, 0x21, 0xd1, 0xd3, 0x89, 0x20, 0x08, 0xed, 0x16,
		0x3c, 0x0c, 0xde, 0x88, 0xa6, 0xd8, 0xd6, 0xd6, 0xee, 0x7a,
		0xa2, 0x51, 0x1e, 0x15, 0xd5, 0x83, 0xe3, 0xf2, 0x86, 0x46,
		0x2f, 0x9d, 0x3d, 0x13, 0x96, 0x6f, 0x95, 0x58, 0x98, 0xd0,
		0xc0, 0xc7, 0xa3, 0x69, 0xe8, 0x86, 0xca, 0xf2, 0xc3, 0xdf,
		0x6d, 0x41, 0x23, 0x8b, 0xa8, 0x50, 0x86, 0xe4, 0x7f, 0xa7,
		0x80, 0x28, 0xf3, 0xdc, 0x01, 0x54, 0x58, 0x34, 0xdc, 0x73,
		0xb2, 0x55, 0x09, 0xee, 0x65, 0x77, 0xa6, 0xe8, 0xf1, 0x59,
		0xc7, 0x04, 0x07, 0xbb, 0x7b, 0x05, 0xca, 0x9b, 0xfe, 0x25,
		0x80, 0xe8, 0xab, 0x86, 0x45, 0x43, 0xfb, 0xe6, 0x7c, 0x28,
		0x62, 0x30, 0xdf, 0xb3, 0xc7, 0x92, 0xad, 0x05, 0x0e, 0x79,
		0xce, 0x16, 0xca, 0x3f, 0xab, 0xf7, 0x33, 0x87, 0xe8, 0x5a,
		0x68, 0xfa, 0xa4, 0x4e, 0xe9, 0x5d
	};
	const uint8_t entropy_9[] = { 0x09, 0xbc, 0x30, 0x30, 0xbe, 0x92, 0x04,
				      0x35, 0xcf, 0x5d, 0x5c, 0x81, 0x3f, 0xb2,
				      0x50, 0xbe, 0x9a, 0xc3, 0x28, 0x72, 0xc8,
				      0x3d, 0x96, 0x35, 0xdd, 0xcc, 0x38, 0x29,
				      0x13, 0x7d, 0x1d, 0x9a };
	const uint8_t nonce_9[] = { 0x43, 0xe5, 0x77, 0xbb, 0xdc, 0x08,
				    0x4d, 0x4d, 0x03, 0x32, 0xc3, 0x11,
				    0x04, 0xeb, 0x2e, 0x4b };
	const uint8_t personalization_string_9[] = {
		0x39, 0xf8, 0xde, 0x35, 0x16, 0x1a, 0x25, 0x09,
		0xa4, 0x07, 0x7c, 0xf9, 0xf9, 0x2e, 0xb6, 0x53,
		0x60, 0x38, 0x20, 0xdb, 0x6f, 0xa8, 0x64, 0x91,
		0x55, 0x3c, 0xae, 0xeb, 0x9a, 0xd4, 0x6c, 0xff
	};
	const uint8_t returned_bits_9[] = {
		0x9d, 0x5e, 0xd4, 0xdf, 0xf9, 0x96, 0x2d, 0x28, 0x0c, 0x18,
		0xeb, 0x64, 0x99, 0x64, 0xc9, 0x60, 0xc8, 0x97, 0xca, 0x19,
		0x98, 0x02, 0xc9, 0x18, 0x62, 0x45, 0xf9, 0x87, 0xae, 0xa9,
		0x7d, 0x9a, 0x62, 0x31, 0x2e, 0xa5, 0xd8, 0xf6, 0xa6, 0x76,
		0xf6, 0xc7, 0x2f, 0xeb, 0xde, 0xe2, 0x32, 0x25, 0x7b, 0xc2,
		0xbd, 0x40, 0x07, 0x5a, 0xf9, 0x35, 0xd7, 0xdc, 0x62, 0xd9,
		0x88, 0xc8, 0xc0, 0x3f, 0x6d, 0x27, 0x6b, 0x1e, 0x8b, 0x74,
		0x49, 0x61, 0x88, 0xee, 0x82, 0x3c, 0xd4, 0x4b, 0xad, 0x8b,
		0xb0, 0x87, 0xdf, 0x4c, 0xb7, 0xc9, 0x0b, 0x94, 0xb9, 0x34,
		0x40, 0xbf, 0x5d, 0xcc, 0xbd, 0x11, 0x19, 0x37, 0x22, 0x16,
		0x98, 0xc1, 0x46, 0x93, 0xd2, 0xa1, 0x23, 0x65, 0x13, 0xdf,
		0x27, 0x11, 0x40, 0xbc, 0xf8, 0xaf, 0x22, 0xb5, 0x88, 0x4e,
		0x3d, 0x11, 0x68, 0x47, 0xc6, 0x3d, 0xce, 0x91, 0x01, 0xe2,
		0x76, 0x03, 0xd4, 0x8f, 0x8d, 0x94, 0x42, 0x1b, 0x92, 0xaa,
		0x38, 0x54, 0x7d, 0x12, 0xfd, 0x7a, 0xa4, 0x90, 0x26, 0x7d,
		0xbb, 0xb7, 0xca, 0x14, 0x2a, 0xb3, 0x92, 0xf4, 0xf0, 0xa5,
		0x95, 0x53, 0xba, 0x95, 0xba, 0x1c, 0x37, 0x0c, 0x2a, 0x55,
		0xaf, 0x5d, 0x63, 0xcf, 0xce, 0x6b, 0xb6, 0x18, 0x29, 0x45,
		0xea, 0xaf, 0xc5, 0x37, 0x6c, 0x11, 0x7b, 0x06, 0xe9, 0x9a,
		0x4f, 0x1e, 0xcc, 0x36, 0xa0, 0xa8, 0x68, 0x2d, 0x66, 0xe9,
		0xea, 0xdf, 0xfc, 0x15, 0x67, 0xb7, 0x58, 0xee, 0xc1, 0xb4,
		0xe3, 0xbd, 0xed, 0x98, 0x2c, 0x09, 0x7a, 0xef, 0x4f, 0xb5,
		0x79, 0x9c, 0xc6, 0x59, 0xa2, 0x62, 0x3f, 0xfc, 0x8f, 0xb6,
		0x9f, 0x52, 0x58, 0x08, 0x3d, 0x42, 0x32, 0x43, 0x44, 0xe6,
		0x15, 0x09, 0xaa, 0x29, 0xce, 0xa4, 0x8c, 0xc3, 0xc3, 0xe1,
		0x44, 0x9b, 0x01, 0xcf, 0x77, 0x32
	};
	const uint8_t additional_input_1_9[] = {
		0x3d, 0x97, 0xfd, 0x3f, 0x3b, 0xc2, 0x7d, 0x28,
		0x06, 0xa4, 0x41, 0xe0, 0xc3, 0x00, 0xf3, 0x6e,
		0x04, 0x54, 0x2a, 0x4b, 0x9c, 0x34, 0x64, 0x21,
		0x17, 0x67, 0x67, 0x0e, 0xba, 0x8f, 0x97, 0x0b
	};
	const uint8_t additional_input_2_9[] = {
		0xbb, 0xb6, 0x5d, 0xf7, 0x91, 0xd2, 0x10, 0x0a,
		0x6b, 0x81, 0xb3, 0x5f, 0x23, 0xc2, 0x7c, 0x1e,
		0xd3, 0x74, 0xde, 0x5c, 0xf6, 0x15, 0x7c, 0xe9,
		0xba, 0xd2, 0x83, 0x4d, 0x2a, 0x75, 0x7c, 0x63
	};

	static struct nist_test_vector_hmac_drbg test_vectors[] = {
		HMAC_TEST_VECTOR(1),	HMAC_TEST_VECTOR(2),
		HMAC_TEST_VECTOR(3),	HMAC_TEST_VECTOR(4),
		HMAC_TEST_VECTOR(5),	HMAC_TEST_VECTOR(6),
		HMAC_TEST_VECTOR(7),	HMAC_TEST_VECTOR(8),
		HMAC_TEST_VECTOR_AI(9),
	};

	for (size_t i = 0;
	     i < sizeof(test_vectors) / sizeof(nist_test_vector_hmac_drbg);
	     ++i) {
		const size_t entropy_size = 32;
		const size_t nonce_size = 16;
		const size_t pers_size = 32;
		const size_t result_size = 256;
		const size_t additonal_input_len = 32;
		size_t seed_material_size =
			entropy_size + nonce_size + pers_size;
		size_t offset = 0;

		seed_material = (uint8_t *)malloc(seed_material_size);
		act = (uint8_t *)malloc(result_size);

		memcpy(seed_material + offset, test_vectors[i].entropy,
		       entropy_size);
		offset += entropy_size;
		memcpy(seed_material + offset, test_vectors[i].nonce,
		       nonce_size);
		offset += nonce_size;
		memcpy(seed_material + offset,
		       test_vectors[i].personalization_string, pers_size);
		offset += pers_size;

		CKINT(esdm_botan_drbg_alloc(&drng, 256));
		CKINT(esdm_botan_drbg_seed(drng, seed_material,
					   seed_material_size));
		if (esdm_botan_drbg_generate_w_additional_data(
			    drng, act, result_size,
			    test_vectors[i].additional_input_1,
			    test_vectors[i].additional_input_1 ?
				    additonal_input_len :
				    0) != (ssize_t)result_size) {
			ret = -EFAULT;
			goto out;
		}
		if (esdm_botan_drbg_generate_w_additional_data(
			    drng, act, result_size,
			    test_vectors[i].additional_input_2,
			    test_vectors[i].additional_input_2 ?
				    additonal_input_len :
				    0) != (ssize_t)result_size) {
			ret = -EFAULT;
			goto out;
		}
		if (memcmp(act, test_vectors[i].returned_bits, result_size)) {
			ret = -EFAULT;
			goto out;
		}
		esdm_botan_drbg_dealloc(drng);
		drng = NULL;

		free(seed_material);
		seed_material = NULL;

		free(act);
		act = NULL;
	}

out:
	esdm_botan_drbg_dealloc(drng);

	/* does nothing if already NULL */
	free(seed_material);
	seed_material = NULL;

	/* does nothing if already NULL */
	free(act);
	act = NULL;

	return ret;
}
#endif

static int esdm_botan_drbg_selftest(void)
{
#ifdef ESDM_BOTAN_DRNG_CHACHA20
	return esdm_botan_drbg_selftest_chacha20();
#endif

#ifdef ESDM_BOTAN_DRNG_HMAC
	return esdm_botan_drbg_selftest_hmac();
#endif
}

const struct esdm_drng_cb esdm_botan_drbg_cb = {
	.drng_name = esdm_botan_drbg_name,
	.drng_selftest = esdm_botan_drbg_selftest,
	.drng_alloc = esdm_botan_drbg_alloc,
	.drng_dealloc = esdm_botan_drbg_dealloc,
	.drng_seed = esdm_botan_drbg_seed,
	.drng_generate = esdm_botan_drbg_generate,
};
