/*
 * Backend for the ESDM providing a ChaCha20-based DRNG
 *
 * Copyright (C) 2022 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include <errno.h>

#include "esdm_crypto.h"
#include "lc_hash_drbg_sha512.h"
#include "esdm_builtin_hash_drbg.h"
#include "esdm_logger.h"

static int esdm_hash_drbg_seed(void *drng, const uint8_t *inbuf,
			       size_t inbuflen)
{
	struct lc_drbg_state *drbg = (struct lc_drbg_state *)drng;

	return lc_drbg_seed(drbg, inbuf, inbuflen, NULL, 0);
}

static ssize_t esdm_hash_drbg_generate(void *drng, uint8_t *outbuf,
				       size_t outbuflen)
{
	struct lc_drbg_state *drbg = (struct lc_drbg_state *)drng;

	return lc_drbg_generate(drbg, outbuf, outbuflen, NULL, 0);
}

static int esdm_hash_drbg_alloc(void **drng, uint32_t sec_strength)
{
	struct lc_drbg_state **drbg = (struct lc_drbg_state **)drng;
	int ret = lc_drbg_hash_alloc(drbg);

	(void)sec_strength;

	if (ret < 0)
		return ret;

	esdm_logger(LOGGER_VERBOSE, LOGGER_C_ANY, "Hash DRBG core allocated\n");

	return ret;
}

static void esdm_hash_drbg_dealloc(void *drng)
{
	struct lc_drbg_state *drbg = (struct lc_drbg_state *)drng;

	lc_drbg_zero_free(drbg);
	esdm_logger(LOGGER_VERBOSE, LOGGER_C_ANY,
		    "Hash DRBG core zeroized and freed\n");
}

static const char *esdm_hash_drbg_name(void)
{
	return "builtin SP800-90A Hash DRBG";
}

static int esdm_hash_drbg_selftest(void)
{
	static const uint8_t ent_nonce[] = {
		0x9E, 0x28, 0x52, 0xF1, 0xD8, 0xB2, 0x3C, 0x1A, 0x80, 0xCA,
		0x75, 0x29, 0x37, 0xAC, 0x58, 0x54, 0x61, 0x98, 0xDB, 0x72,
		0x81, 0xB7, 0x43, 0xDB, 0x37, 0x21, 0x8E, 0x86, 0x40, 0x3B,
		0x74, 0xF9, 0x88, 0x45, 0x49, 0xDC, 0x49, 0x26, 0xBB, 0xAA,
		0x83, 0x3E, 0x50, 0x42, 0xA9, 0x52, 0xAE, 0x97, 0xB2, 0x1B,
		0x72, 0x93, 0x7C, 0xC7, 0x29, 0x5C, 0x47, 0x2B, 0x70, 0xFB,
		0xEC, 0xAC, 0xD9, 0x2C
	};
	static const uint8_t pers[] = { 0x12, 0x6B, 0xE1, 0x49, 0x3F, 0x41,
					0x28, 0x9A, 0xDC, 0x5C, 0x7F, 0x00,
					0x43, 0x40, 0xFF, 0x21, 0xA7, 0xEC,
					0x4D, 0xAD, 0xFF, 0xDA, 0x64, 0x2D,
					0xE4, 0x65, 0xAB, 0x2E, 0x98, 0x54,
					0x19, 0x1A };
	static const uint8_t addtl1[] = { 0x89, 0x18, 0x8A, 0xB5, 0x82, 0x0B,
					  0x05, 0x98, 0xF9, 0x81, 0xB3, 0x34,
					  0x44, 0x6D, 0xD4, 0x38, 0x29, 0xCD,
					  0x50, 0x4E, 0x06, 0xFE, 0x11, 0xF2,
					  0x3C, 0x70, 0x0D, 0xAC, 0xA8, 0x28,
					  0x0E, 0x40 };
	static const uint8_t addtl2[] = { 0x67, 0x87, 0xEE, 0x02, 0xA6, 0x0F,
					  0x2F, 0x8D, 0x8D, 0xF3, 0x4A, 0xBF,
					  0xA3, 0x61, 0x7E, 0xD6, 0xB2, 0xB1,
					  0x37, 0x61, 0xA5, 0x41, 0xB3, 0x8C,
					  0x2A, 0xF9, 0x01, 0x08, 0x3F, 0xC9,
					  0x0D, 0xCA };
	static const uint8_t exp[] = {
		0x4d, 0xa6, 0x34, 0x92, 0x52, 0x48, 0x31, 0x53, 0x5c, 0x2d,
		0xd8, 0xe9, 0xbd, 0x2f, 0x31, 0x9b, 0x11, 0xc2, 0xda, 0x2f,
		0xd7, 0x21, 0x05, 0xed, 0x2c, 0x67, 0x04, 0x37, 0xbd, 0x53,
		0xb3, 0x4e, 0x9d, 0x0c, 0x16, 0x54, 0x89, 0xca, 0xe3, 0x39,
		0xc0, 0x77, 0xb3, 0xb5, 0xfa, 0xae, 0x9c, 0x59, 0x90, 0x43,
		0x09, 0x43, 0xf1, 0x4c, 0x70, 0x3e, 0x00, 0x02, 0xa7, 0xf3,
		0x13, 0x93, 0x98, 0xba, 0x8b, 0xf4, 0xdf, 0x9e, 0x3f, 0x8d,
		0x65, 0x0f, 0x7a, 0x35, 0xd7, 0xa1, 0x4d, 0x13, 0x70, 0x50,
		0x01, 0xd8, 0x54, 0x26, 0x74, 0x2a, 0xdc, 0x35, 0xb6, 0x59,
		0xc2, 0xfb, 0x75, 0xfa, 0x47, 0x7c, 0x06, 0x26, 0xfc, 0xcc,
		0x20, 0xa0, 0x11, 0xc4, 0xc4, 0xe8, 0xe5, 0x79, 0x33, 0x39,
		0x30, 0x64, 0xb3, 0x75, 0x7b, 0x2f, 0x04, 0x52, 0x0a, 0x60,
		0x41, 0x71, 0xcf, 0x3b, 0x1f, 0x30, 0x5b, 0x81, 0x53, 0x2a,
		0x26, 0xde, 0x3a, 0x4c, 0x5a, 0x64, 0xe2, 0x29, 0x3e, 0x38,
		0x8f, 0x8e, 0x1e, 0x76, 0x08, 0xea, 0x81, 0x9e, 0x5d, 0x7b,
		0x3a, 0xad, 0x64, 0xc7, 0x1c, 0x32, 0x51, 0x9d, 0x67, 0xe3,
		0x75, 0x8f, 0x73, 0x23, 0x55, 0xbd, 0x1b, 0x70, 0x9a, 0x8b,
		0x8f, 0x5d, 0xcf, 0xe5, 0xac, 0x6d, 0xc9, 0xf9, 0x48, 0xfc,
		0xeb, 0xd6, 0x3a, 0x37, 0x01, 0x4e, 0x6a, 0xae, 0x7b, 0x83,
		0xf5, 0x13, 0x22, 0x97, 0x2b, 0xc8, 0xd0, 0x9d, 0xd4, 0x91,
		0x18, 0xa1, 0x4b, 0x36, 0xf3, 0x0d, 0x3f, 0x4e, 0x6d, 0x96,
		0x8d, 0x79, 0xd8, 0xd7, 0xf0, 0x31, 0x57, 0xf8, 0x32, 0x93,
		0x10, 0xf6, 0xba, 0xab, 0x57, 0xa6, 0xec, 0xb8, 0xbc, 0x9b,
		0x0b, 0xef, 0xa5, 0x00, 0x78, 0x7f, 0x63, 0x3e, 0x0f, 0x45,
		0x3b, 0x6d, 0xd9, 0xea, 0x58, 0xee, 0x29, 0x48, 0xad, 0x33,
		0xcb, 0x1b, 0xbf, 0xd1, 0x1d, 0x2a
	};
	uint8_t act[256];
	LC_DRBG_HASH_CTX_ON_STACK(drbg_stack);
	int ret = -EFAULT;

	if (lc_drbg_healthcheck_sanity(drbg_stack))
		goto out;

	if (lc_drbg_seed(drbg_stack, ent_nonce, 64, pers, 32))
		goto out;

	if (lc_drbg_generate(drbg_stack, act, 256, addtl1, 32) < 0)
		goto out;

	if (lc_drbg_generate(drbg_stack, act, 256, addtl2, 32) < 0)
		goto out;

	if (!memcmp(act, exp, 256))
		ret = 0;

out:
	lc_drbg_zero(drbg_stack);
	return ret;
}

const struct esdm_drng_cb esdm_builtin_hash_drbg_cb = {
	.drng_name = esdm_hash_drbg_name,
	.drng_selftest = esdm_hash_drbg_selftest,
	.drng_alloc = esdm_hash_drbg_alloc,
	.drng_dealloc = esdm_hash_drbg_dealloc,
	.drng_seed = esdm_hash_drbg_seed,
	.drng_generate = esdm_hash_drbg_generate,
};
